<?php

namespace Cite\Tests\Integration;

use Cite\AnchorFormatter;
use Cite\ReferenceMessageLocalizer;
use Message;
use Wikimedia\TestingAccessWrapper;

/**
 * @coversDefaultClass \Cite\AnchorFormatter
 *
 * @license GPL-2.0-or-later
 */
class AnchorFormatterTest extends \MediaWikiIntegrationTestCase {

	protected function setUp(): void {
		parent::setUp();

		global $wgFragmentMode;
		$wgFragmentMode = [ 'html5' ];
	}

	/**
	 * @covers ::refKey
	 */
	public function testRefKey() {
		$mockMessageLocalizer = $this->createMock( ReferenceMessageLocalizer::class );
		$mockMessageLocalizer->method( 'msg' )->willReturnCallback(
			function ( ...$args ) {
				$msg = $this->createMock( Message::class );
				$msg->method( 'plain' )->willReturn( '(plain:' . implode( '|', $args ) . ')' );
				$msg->method( 'text' )->willReturn( '(text:' . implode( '|', $args ) . ')' );
				return $msg;
			}
		);
		$formatter = new AnchorFormatter( $mockMessageLocalizer );

		$this->assertSame(
			'(text:cite&#95;reference&#95;link&#95;prefix)key(text:cite&#95;reference&#95;link&#95;suffix)',
			$formatter->refKey( 'key', null ) );
		$this->assertSame(
			'(text:cite&#95;reference&#95;link&#95;prefix)' .
				// phpcs:ignore Generic.Files.LineLength.TooLong
				'(plain:cite&#95;reference&#95;link&#95;key&#95;with&#95;num&#124;key&#124;2)(text:cite&#95;reference&#95;link&#95;suffix)',
			$formatter->refKey( 'key', '2' ) );
	}

	/**
	 * @covers ::getReferencesKey
	 */
	public function testGetReferencesKey() {
		$mockMessageLocalizer = $this->createMock( ReferenceMessageLocalizer::class );
		$mockMessageLocalizer->method( 'msg' )->willReturnCallback(
			function ( ...$args ) {
				$msg = $this->createMock( Message::class );
				$msg->method( 'text' )->willReturn( '(' . implode( '|', $args ) . ')' );
				return $msg;
			}
		);
		$formatter = new AnchorFormatter( $mockMessageLocalizer );

		$this->assertSame(
			'(cite&#95;references&#95;link&#95;prefix)key(cite&#95;references&#95;link&#95;suffix)',
			$formatter->getReferencesKey( 'key' ) );
	}

	/**
	 * @covers ::normalizeKey
	 * @covers ::__construct
	 * @dataProvider provideKeyNormalizations
	 */
	public function testNormalizeKey( $key, $expected ) {
		/** @var AnchorFormatter $formatter */
		$formatter = TestingAccessWrapper::newFromObject( new AnchorFormatter(
			$this->createMock( ReferenceMessageLocalizer::class ) ) );
		$this->assertSame( $expected, $formatter->normalizeKey( $key ) );
	}

	public function provideKeyNormalizations() {
		return [
			[ 'a b', 'a&#95;b' ],
			[ 'a  __  b', 'a&#95;b' ],
			[ ':', ':' ],
			[ "\t\n", '&#95;' ],
			[ "'", '&#039;' ],
			[ "''", '&#039;&#039;' ],
			[ '"%&/<>?[]{|}', '&quot;%&amp;/&lt;&gt;?&#91;&#93;&#123;&#124;&#125;' ],
			[ 'ISBN', '&#73;SBN' ],

			[ 'nature%20phylo', 'nature%2520phylo' ],
			[ 'Mininova%2E26%2E11%2E2009', 'Mininova%252E26%252E11%252E2009' ],
			[ '%c8%98tiri_2019', '%25c8%2598tiri&#95;2019' ],
			[ 'play%21', 'play%2521' ],
			[ 'Terry+O%26rsquo%3BN…</ref', 'Terry+O%2526rsquo%253BN…&lt;/ref' ],
			[ '9&nbsp;pm', '9&amp;nbsp;pm' ],
			[ 'n%25%32%30n', 'n%2525%2532%2530n' ],
			[ 'a_ %20a', 'a&#95;%2520a' ],
		];
	}

}
