;;; Guile-Git --- GNU Guile bindings of libgit2
;;; Copyright © 2025 Ludovic Courtès <ludo@gnu.org>
;;;
;;; This file is part of Guile-Git.
;;;
;;; Guile-Git is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; Guile-Git is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with Guile-Git.  If not, see <http://www.gnu.org/licenses/>.

(define-module (tests graph)
  #:use-module (git)
  #:use-module (git structs)
  #:use-module (tests helpers)
  #:use-module (srfi srfi-64)
  #:use-module (rnrs bytevectors)
  #:use-module (rnrs io ports)
  #:use-module (ice-9 match))

(test-begin "graph")

(with-repository "simple" directory
  (test-equal "graph-ahead-behind"
    '((0 1)                                       ;1 behind
      (1 0))                                      ;1 ahead
    (let* ((repository (repository-open directory))
           (head (reference-target (repository-head repository)))
           (parent (car (commit-parents (commit-lookup repository head)))))
      (list (call-with-values
                (lambda () (graph-ahead-behind repository
                                               (commit-id parent)
                                               head))
              list)
            (call-with-values
                (lambda () (graph-ahead-behind repository
                                               head
                                               (commit-id parent)))
              list))))

  (test-equal "graph-ahead-behind, unknown OID"
    'git-error
    (let* ((repository (repository-open directory))
           (head (reference-target (repository-head repository)))
           (oid (string->oid "0000000000000000000000000000000000000042")))
      (catch 'git-error
        (lambda ()
          (graph-ahead-behind repository oid head))
        (lambda (key error . _)
          ;; The exact error code we should get is unspecified.
          (pk 'error error key)))))

  (test-assert "graph-descendant?"
    (let* ((repository (repository-open directory))
           (head (reference-target (repository-head repository)))
           (parent (car (commit-parents (commit-lookup repository head)))))
      (and (not (graph-descendant? repository head head))
           (let loop ((commit (commit-lookup repository head)))
             (match (commit-parents commit)
               (()
                #t)
               ((parent)
                (pk 'descendant? head (commit-id parent))
                (and (graph-descendant? repository
                                        head (commit-id parent))
                     (not (graph-descendant? repository
                                             (commit-id parent) head))
                     (loop parent))))))))

  (test-equal "graph-descendant?, unknown OID"
    'git-error
    (let* ((repository (repository-open directory))
           (head (reference-target (repository-head repository)))
           (oid (string->oid "0000000000000000000000000000000000000042")))
      (catch 'git-error
        (lambda ()
          (graph-descendant? repository head oid))
        (lambda (key error . _)
          ;; The exact error code we should get is unspecified.
          (pk 'error error key))))))

(test-end)
