use ui;
use progvis:net;

on Ui:

/**
 * Data for the entire Problem dialog. Allows asking for different types of problems.
 */
class ProblemData {
	private Str[] names;
	private View[] views;

	init() {
		init {}
	}

	void add(Str name, View view) {
		names << name;
		views << view;
	}

	Nat count() { names.count; }
	Str name(Nat id) { names[id]; }
	View view(Nat id) { views[id]; }
	Str[] allNames() { names; }
}

/**
 * Data for some view. Controls what kind of information shall be displayed.
 *
 * Does not actually store any data, but encapsulates the queries that need to be done to acquire the data.
 */
class View {
	protected Client client;

	init(Client client) {
		init { client = client; }
	}

	value Row {
		// Some ID used to refer to this row.
		Int id;

		// Data shown to the user.
		Str[] data;

		init(Int id, Str[] data) {
			init { id = id; data = data; }
		}
	}

	Str[] labels() : abstract;
	Row[] rows() : abstract;

	protected Str[] problemLabels() {
		["ID", "Problem name", "Author", "Type", "Improved"];
	}

	protected Row[] getProblems(Request req) {
		if (ans = client.query(req) as ProblemResponse) {
			ans.problems.sort();
			Row[] data;
			for (r in ans.problems) {
				data << Row(r.id, [r.id.toS, r.title, r.author, r.typeStr, r.improvedStr]);
			}
			return data;
		} else {
			return [];
		}
	}

	// Find an appropriate panel for when a row has been clicked.
	Panel details(Int id) : abstract;
}

/**
 * View for problems that we can solve (e.g. unsolved problems).
 */
class UnsolvedView extends View {
	private Callback cb;
	private Bool currentError;

	init(Client client, Callback cb, Bool currentError) {
		init(client) {
			cb = cb;
			currentError = currentError;
		}
	}

	Str[] labels() : override {
		problemLabels();
	}

	Row[] rows() : override {
		getProblems(UnsolvedRequest(currentError));
	}

	Panel details(Int id) : override {
		ProblemInfo? parent;
		if (response = client.query(StatsRequest(id, false, false)) as StatsResponse) {
			parent = response.parent;
		}

		StatsPanel(id, parent, true, cb);
	}
}

/**
 * View for problems that we have solved.
 */
class SolvedView extends View {
	private Callback cb;

	init(Client client, Callback cb) {
		init(client) {
			cb = cb;
		}
	}

	Str[] labels() : override {
		problemLabels();
	}

	Row[] rows() : override {
		getProblems(SolvedProblemsRequest());
	}

	Panel details(Int id) : override {
		if (response = client.query(StatsRequest(id, true, true)) as StatsResponse) {
			MultiPanel root;
			root.add(StatsPanel(id, response.parent, false, cb));
			if (response.solutions.any)
				root.add(TablePanel(SolutionView(client, response.solutions, cb)));
			root.add(TablePanel(ImprovementView(client, response.improvements, cb)));
			root;
		} else {
			// Good for zero height.
			MultiPanel();
		}
	}
}

/**
 * View for own problems.
 */
class OwnView extends SolvedView {
	init(Client client, Callback cb) {
		init(client, cb) {}
	}

	Row[] rows() : override {
		getProblems(OwnProblemsRequest());
	}
}


/**
 * View for solutions.
 */
class SolutionView extends View {
	private Solution[] solutions;
	private Callback cb;

	init(Client client, Solution[] solutions, Callback cb) {
		init(client) { solutions = solutions; cb = cb; }
		solutions.sort((a, b) => a.problemId < b.problemId);
	}

	Str[] labels() : override {
		["Author", "Type of error found"];
	}

	Row[] rows() : override {
		Row[] result;
		for (r in solutions) {
			result << Row(r.id, [r.author, r.type]);
		}
		result;
	}

	Panel details(Int id) : override {
		for (x in solutions) {
			if (x.id == id) {
				return SolutionPanel(x, cb);
			}
		}

		// Good for "zero height".
		return MultiPanel();
	}
}


/**
 * View for improvements.
 */
class ImprovementView extends View {
	private ProblemInfo[] improvements;
	private Callback cb;

	init(Client client, ProblemInfo[] improvements, Callback cb) {
		init(client) { improvements = improvements; cb = cb; }
		improvements.sort();
	}

	Str[] labels() : override {
		["ID", "Improvement title", "Author", "Type"];
	}

	Row[] rows() : override {
		Row[] result;
		for (r in improvements) {
			result << Row(r.id, [r.id.toS, r.title, r.author, r.typeStr]);
		}
		result;
	}

	Panel details(Int id) : override {
		ProblemInfo? parent;
		Bool solvable;
		if (response = client.query(StatsRequest(id, false, false)) as StatsResponse) {
			parent = response.parent;
			solvable = !response.solved;
		}

		return StatsPanel(id, parent, solvable, cb);
	}
}
