# Copyright (c) 2011-2012 OpenStack Foundation
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from oslo_log import log as logging

import nova.conf
from nova.scheduler import filters
from nova.scheduler.filters import utils

LOG = logging.getLogger(__name__)

CONF = nova.conf.CONF


class AvailabilityZoneFilter(filters.BaseHostFilter):
    """Filters Hosts by availability zone.

    Works with aggregate metadata availability zones, using the key
    'availability_zone'
    Note: in theory a compute node can be part of multiple availability_zones
    """

    # Availability zones do not change within a request
    run_filter_once_per_request = True

    RUN_ON_REBUILD = False

    def __init__(self):
        super().__init__()
        if CONF.scheduler.query_placement_for_availability_zone:
            LOG.warning(
                "The 'AvailabilityZoneFilter' is deprecated since the 24.0.0 "
                "(Xena) release. Since the 18.0.0 (Rocky) release, nova "
                "has supported mapping AZs to placement aggregates. "
                "The feature is enabled by the "
                "'query_placement_for_availability_zone' config option and "
                "is now enabled by default. As such, the "
                "'AvailabilityZoneFilter' is no longer required. Nova is "
                "currently configured to use both placement and the "
                "AvailabilityZoneFilter for AZ enforcement."
            )

    def host_passes(self, host_state, spec_obj):
        availability_zone = spec_obj.availability_zone

        if not availability_zone:
            return True

        metadata = utils.aggregate_metadata_get_by_host(
                host_state, key='availability_zone')

        if 'availability_zone' in metadata:
            hosts_passes = availability_zone in metadata['availability_zone']
            host_az = metadata['availability_zone']
        else:
            hosts_passes = availability_zone == CONF.default_availability_zone
            host_az = CONF.default_availability_zone

        if not hosts_passes:
            LOG.debug("Availability Zone '%(az)s' requested. "
                      "%(host_state)s has AZs: %(host_az)s",
                      {'host_state': host_state,
                       'az': availability_zone,
                       'host_az': host_az})

        return hosts_passes
